const { z } = require('zod');
const axios = require('axios');
const prisma = require('../prisma');
const { createUser, verifyUser, findOrCreateGoogleUser, setPassword } = require('../services/userService');
const { signToken, verifyToken } = require('../utils/jwt');
const { createError } = require('../utils/errors');

function getGoogleConfig() {
  const clientId = process.env.GOOGLE_CLIENT_ID;
  const clientSecret = process.env.GOOGLE_CLIENT_SECRET;
  const callbackUrl = process.env.GOOGLE_CALLBACK_URL;
  const frontendBaseUrl = process.env.FRONTEND_BASE_URL || 'http://localhost:3000';
  if (!clientId || !clientSecret || !callbackUrl) {
    throw createError(500, 'Google OAuth is not configured');
  }
  return { clientId, clientSecret, callbackUrl, frontendBaseUrl };
}

function buildGoogleAuthUrl(state) {
  const { clientId, callbackUrl } = getGoogleConfig();
  const params = new URLSearchParams({
    client_id: clientId,
    redirect_uri: callbackUrl,
    response_type: 'code',
    scope: 'openid email profile',
    access_type: 'offline',
    prompt: 'select_account',
    state,
  });
  return `https://accounts.google.com/o/oauth2/v2/auth?${params.toString()}`;
}

const registerSchema = z.object({
  name: z.string().min(2),
  email: z.string().email(),
  phone: z.string().optional(),
  password: z.string().min(6),
});

const loginSchema = z.object({
  email: z.string().email(),
  password: z.string().min(6),
});

const forgotSchema = z.object({
  email: z.string().email(),
});

const resetSchema = z.object({
  token: z.string().min(10),
  newPassword: z.string().min(6),
});

async function register(req, res, next) {
  try {
    const payload = registerSchema.parse(req.body);
    const user = await createUser(payload);
    const token = signToken({ sub: user.id, role: user.role });
    res.json({
      token,
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
        phone: user.phone,
      },
    });
  } catch (err) {
    next(err);
  }
}

async function login(req, res, next) {
  try {
    const payload = loginSchema.parse(req.body);
    const user = await verifyUser(payload.email, payload.password);
    const token = signToken({ sub: user.id, role: user.role });
    res.json({
      token,
      user: {
        id: user.id,
        name: user.name,
        email: user.email,
      },
    });
  } catch (err) {
    next(err);
  }
}

async function googleStart(req, res, next) {
  try {
    const { frontendBaseUrl } = getGoogleConfig();
    const redirectTarget = req.query.redirect || frontendBaseUrl;
    const state = signToken({ redirect: redirectTarget }, { expiresIn: '10m' });
    const url = buildGoogleAuthUrl(state);
    res.redirect(url);
  } catch (err) {
    next(err);
  }
}

async function googleCallback(req, res, next) {
  try {
    const { code, state } = req.query;
    if (!code || !state) {
      return res.status(400).json({ message: 'Missing OAuth code/state' });
    }
    const { clientId, clientSecret, callbackUrl, frontendBaseUrl } = getGoogleConfig();
    let redirectTarget = frontendBaseUrl;
    try {
      const payload = verifyToken(state);
      if (payload?.redirect) {
        redirectTarget = payload.redirect;
      }
    } catch (err) {
      // ignore invalid state; fallback to default redirect
    }

    const tokenRes = await axios.post('https://oauth2.googleapis.com/token', new URLSearchParams({
      code,
      client_id: clientId,
      client_secret: clientSecret,
      redirect_uri: callbackUrl,
      grant_type: 'authorization_code',
    }).toString(), {
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    });

    const accessToken = tokenRes.data.access_token;
    if (!accessToken) {
      throw createError(400, 'Google access token not received');
    }

    const profileRes = await axios.get('https://openidconnect.googleapis.com/v1/userinfo', {
      headers: { Authorization: `Bearer ${accessToken}` },
    });
    const profile = profileRes.data || {};
    const email = profile.email;
    const name = profile.name || profile.given_name || 'User';
    if (!email) {
      throw createError(400, 'Google account email not available');
    }

    const user = await findOrCreateGoogleUser({ name, email });
    const token = signToken({ sub: user.id, role: user.role });
    const url = new URL(`${redirectTarget.replace(/\/$/, '')}/login.html`);
    url.searchParams.set('token', token);
    res.redirect(url.toString());
  } catch (err) {
    next(err);
  }
}

async function forgotPassword(req, res, next) {
  try {
    const payload = forgotSchema.parse(req.body);
    const existing = await prisma.user.findUnique({ where: { email: payload.email } });
    if (!existing) {
      return res.json({ message: 'If the email exists, a reset link has been sent.' });
    }
    const token = signToken({ sub: existing.id, purpose: 'password_reset' }, { expiresIn: '15m' });
    const isProd = (process.env.NODE_ENV || '').toLowerCase() === 'production';
    if (isProd) {
      return res.json({ message: 'If the email exists, a reset link has been sent.' });
    }
    return res.json({
      message: 'Reset token generated (dev only).',
      reset_token: token,
      reset_url: `${process.env.FRONTEND_BASE_URL || 'http://localhost:3000'}/login.html?reset_token=${encodeURIComponent(token)}`,
    });
  } catch (err) {
    next(err);
  }
}

async function resetPassword(req, res, next) {
  try {
    const payload = resetSchema.parse(req.body);
    let tokenPayload;
    try {
      tokenPayload = verifyToken(payload.token);
    } catch (err) {
      return res.status(400).json({ message: 'Invalid or expired reset token' });
    }
    if (tokenPayload?.purpose !== 'password_reset') {
      return res.status(400).json({ message: 'Invalid reset token' });
    }
    await setPassword(tokenPayload.sub, payload.newPassword);
    res.json({ message: 'Password reset successful' });
  } catch (err) {
    next(err);
  }
}

module.exports = { register, login, googleStart, googleCallback, forgotPassword, resetPassword };
