const axios = require('axios');
const crypto = require('crypto');
const { createError } = require('../utils/errors');

function md5(value) {
  return crypto.createHash('md5').update(value).digest('hex');
}

function isMockMode() {
  const mode = (process.env.APIGAMES_MODE || '').toLowerCase().trim();
  if (mode === 'mock' || mode === 'local' || mode === 'test') return true;
  if (mode === 'real' || mode === 'prod' || mode === 'production') return false;
  return false;
}

function mockGames() {
  return {
    data: [
      {
        code: 'mobile_legends',
        name: 'Mobile Legends',
        icon_url: 'https://cdn-icons-png.flaticon.com/128/588/588292.png',
        input_fields: [
          { name: 'user_id', label: 'User ID', type: 'text', required: true },
          { name: 'server', label: 'Server', type: 'text', required: true },
        ],
      },
      {
        code: 'free_fire',
        name: 'Free Fire',
        icon_url: 'https://cdn-icons-png.flaticon.com/128/888/888879.png',
        input_fields: [
          { name: 'user_id', label: 'User ID', type: 'text', required: true },
        ],
      },
      {
        code: 'genshin_impact',
        name: 'Genshin Impact',
        icon_url: 'https://cdn-icons-png.flaticon.com/128/3398/3398649.png',
        input_fields: [
          { name: 'uid', label: 'UID', type: 'text', required: true },
        ],
      },
    ],
  };
}

function mockProductsByGame(gameCode) {
  const base = [
    {
      product_code: `${gameCode}_small`,
      product_name: 'Small Pack',
      price_rp: 5000,
      icon_url: null,
    },
    {
      product_code: `${gameCode}_medium`,
      product_name: 'Medium Pack',
      price_rp: 15000,
      icon_url: null,
    },
    {
      product_code: `${gameCode}_large`,
      product_name: 'Large Pack',
      price_rp: 30000,
      icon_url: null,
    },
  ];
  return { data: base };
}

function getConfig() {
  const baseUrl = process.env.APIGAMES_BASE_URL || 'https://v1.apigames.id';
  const merchantId = process.env.APIGAMES_MERCHANT_ID;
  const secretKey = process.env.APIGAMES_SECRET_KEY;
  if (!merchantId || !secretKey) {
    throw createError(500, 'APIGAMES credentials are not configured');
  }
  const signatureStyle = (process.env.APIGAMES_SIGNATURE_STYLE || 'colon').toLowerCase();
  const signatureV1 = signatureStyle === 'concat'
    ? md5(`${merchantId}${secretKey}`)
    : md5(`${merchantId}:${secretKey}`);
  return { baseUrl, merchantId, secretKey, signatureV1 };
}

function interpolate(template, params) {
  if (!template) return null;
  return template.replace(/\{(.*?)\}/g, (match, key) => {
    const value = params[key] ?? '';
    if (key === 'base_url') {
      return value;
    }
    return encodeURIComponent(value);
  });
}

async function safeGet(url, context) {
  try {
    const res = await axios.get(url);
    return res.data;
  } catch (err) {
    const status = err.response?.status;
    const data = err.response?.data;
    const message = data?.message || data?.error || data?.msg || err.message || 'Unknown error';
    throw createError(502, `APIGAMES ${context} failed${status ? ` (${status})` : ''}: ${message}`);
  }
}

async function listGames() {
  if (isMockMode()) {
    return mockGames();
  }
  const { baseUrl, merchantId, signatureV1 } = getConfig();
  const template = process.env.APIGAMES_GAMES_URL;
  const url = interpolate(template, { base_url: baseUrl, merchant_id: merchantId, signature: signatureV1 });
  if (!url) {
    throw createError(500, 'APIGAMES_GAMES_URL is not configured');
  }
  return safeGet(url, 'listGames');
}

async function listProductsByGame(gameCode) {
  if (isMockMode()) {
    return mockProductsByGame(gameCode);
  }
  const { baseUrl, merchantId, signatureV1 } = getConfig();
  const template = process.env.APIGAMES_PRODUCTS_URL;
  const url = interpolate(template, {
    base_url: baseUrl,
    merchant_id: merchantId,
    signature: signatureV1,
    game_code: gameCode,
  });
  if (!url) {
    throw createError(500, 'APIGAMES_PRODUCTS_URL is not configured');
  }
  return safeGet(url, 'listProducts');
}

async function checkUsername(gameCode, userId) {
  if (isMockMode()) {
    return { data: { username: `MockUser-${userId}`, game: gameCode } };
  }
  const { baseUrl, merchantId, signatureV1 } = getConfig();
  const url = `${baseUrl}/merchant/${merchantId}/cek-username/${gameCode}?user_id=${encodeURIComponent(userId)}&signature=${signatureV1}`;
  return safeGet(url, 'checkUsername');
}

async function createTransaction({ refId, productCode, destination, serverId }) {
  if (isMockMode()) {
    return {
      data: {
        status: 'Sukses',
        trx_id: `MOCK-${refId}`,
        ref_id: refId,
        produk: productCode,
        tujuan: destination,
        server_id: serverId || null,
      },
    };
  }
  const { baseUrl, merchantId, secretKey } = getConfig();
  const signature = md5(`${merchantId}:${secretKey}:${refId}`);
  const url = `${baseUrl}/v2/transaksi?ref_id=${encodeURIComponent(refId)}&merchant_id=${encodeURIComponent(merchantId)}&produk=${encodeURIComponent(productCode)}&tujuan=${encodeURIComponent(destination)}&signature=${signature}&server_id=${encodeURIComponent(serverId || '')}`;
  return safeGet(url, 'createTransaction');
}

async function checkTransactionStatus(refId) {
  if (isMockMode()) {
    return { data: { status: 'Sukses', ref_id: refId } };
  }
  const { baseUrl, merchantId, secretKey } = getConfig();
  const signature = md5(`${merchantId}:${secretKey}:${refId}`);
  const url = `${baseUrl}/v2/transaksi/status?merchant_id=${encodeURIComponent(merchantId)}&ref_id=${encodeURIComponent(refId)}&signature=${signature}`;
  return safeGet(url, 'checkTransactionStatus');
}

module.exports = {
  listGames,
  listProductsByGame,
  checkUsername,
  createTransaction,
  checkTransactionStatus,
};
