const prisma = require('../prisma');
const apigamesService = require('./apigamesService');

const DEFAULT_FIELDS = {
  mobilelegend: [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
    { name: 'server', label: 'Server', type: 'text', required: true },
  ],
  mobile_legends: [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
    { name: 'server', label: 'Server', type: 'text', required: true },
  ],
  ml: [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
    { name: 'server', label: 'Server', type: 'text', required: true },
  ],
  freefire: [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
  ],
  free_fire: [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
  ],
  genshin: [
    { name: 'uid', label: 'UID', type: 'text', required: true },
  ],
  genshin_impact: [
    { name: 'uid', label: 'UID', type: 'text', required: true },
  ],
};

function normalizeGame(game) {
  const code = (game.code || game.game_code || game.kode || game.slug || '').toString().trim();
  const name = game.name || game.game_name || game.nama || code;
  const iconUrl = game.icon || game.icon_url || game.image || game.logo || null;
  const inputFields = game.input_fields || game.inputFields || game.fields || DEFAULT_FIELDS[code] || [
    { name: 'user_id', label: 'User ID', type: 'text', required: true },
  ];
  return { code, name, iconUrl, inputFields };
}

function extractArray(payload) {
  if (Array.isArray(payload)) return payload;
  if (payload && Array.isArray(payload.data)) return payload.data;
  if (payload && Array.isArray(payload.games)) return payload.games;
  return [];
}

async function syncGames() {
  const payload = await apigamesService.listGames();
  const list = extractArray(payload);
  if (!list.length) {
    return [];
  }
  const results = [];
  for (const item of list) {
    const normalized = normalizeGame(item);
    if (!normalized.code) {
      continue;
    }
    const saved = await prisma.game.upsert({
      where: { code: normalized.code },
      update: {
        name: normalized.name,
        iconUrl: normalized.iconUrl,
        inputFields: normalized.inputFields,
      },
      create: {
        code: normalized.code,
        name: normalized.name,
        iconUrl: normalized.iconUrl,
        inputFields: normalized.inputFields,
      },
    });
    results.push(saved);
  }
  return results;
}

async function listGames() {
  const games = await prisma.game.findMany({ where: { enabled: true } });
  if (games.length) {
    return games;
  }
  return syncGames();
}

async function getGameByCode(code) {
  return prisma.game.findUnique({ where: { code } });
}

module.exports = {
  listGames,
  syncGames,
  getGameByCode,
};
