const prisma = require('../prisma');
const productService = require('./productService');
const gameService = require('./gameService');
const apigamesService = require('./apigamesService');
const cashifyService = require('./cashifyService');
const { getSettings } = require('./settingsService');
const { applyPercent, toDecimal } = require('../utils/money');
const { createError } = require('../utils/errors');
const walletService = require('./walletService');

function extractTransactionId(payload) {
  return payload?.transactionId || payload?.data?.transactionId || payload?.data?.transaction_id || payload?.data?.id || null;
}

function extractPaymentMeta(payload) {
  return {
    raw: payload,
    qrisString: payload?.data?.qris_string || payload?.data?.qr_string || payload?.data?.qrcode || payload?.qris_string,
    qrisUrl: payload?.data?.qr_url || payload?.data?.qr_url_string || payload?.qr_url,
    expiredAt: payload?.data?.expired_at || payload?.data?.expiredAt || payload?.data?.expired_time || null,
  };
}

async function createOrder({ user, gameCode, productCode, inputs, paymentMethod }) {
  const settings = await getSettings();
  const products = await productService.listProducts(gameCode);
  const product = products.find((item) => item.code === productCode);
  if (!product) {
    throw createError(404, 'Product not found');
  }
  const priceApi = toDecimal(product.price || 0);
  const margin = applyPercent(priceApi, settings.productMarginPercent || 0);
  const priceSell = priceApi.add(margin);
  const fee = paymentMethod === 'WALLET' ? toDecimal(0) : applyPercent(priceSell, settings.depositFeePercent || 0);
  const totalPay = priceSell.add(fee);

  const game = await gameService.getGameByCode(gameCode);

  if (paymentMethod === 'WALLET') {
    return prisma.$transaction(async (tx) => {
      const wallet = await walletService.ensureSufficientBalance(user.wallet.id, totalPay);
      await walletService.adjustWalletBalance(tx, wallet.id, totalPay.mul(-1));
      await walletService.createWalletTransaction(tx, {
        walletId: wallet.id,
        userId: user.id,
        type: 'PURCHASE',
        status: 'PAID',
        amount: priceSell,
        fee,
        total: totalPay,
        paymentMethod,
        description: `Topup ${product.name}`,
      });
      const order = await tx.topupOrder.create({
        data: {
          userId: user.id,
          gameId: game?.id || null,
          gameCode,
          productCode: product.code,
          productName: product.name,
          priceApi,
          priceSell,
          fee,
          totalPay,
          paymentMethod,
          status: 'PROCESSING',
          targetUserId: inputs.user_id || inputs.uid || inputs.id || '',
          targetServer: inputs.server || null,
          inputPayload: inputs,
        },
      });
      return order;
    }).then(async (order) => {
      try {
        const destination = inputs.user_id || inputs.uid || inputs.id;
        const serverId = inputs.server || '';
        const res = await apigamesService.createTransaction({
          refId: order.id,
          productCode: order.productCode,
          destination,
          serverId,
        });
        const status = res?.data?.status || res?.status || 'Pending';
        await prisma.topupOrder.update({
          where: { id: order.id },
          data: {
            status: status.toLowerCase() === 'sukses' ? 'SUCCESS' : 'PROCESSING',
            supplierRef: res?.data?.trx_id || res?.data?.ref_id || null,
            supplierStatus: status,
          },
        });
      } catch (err) {
        await prisma.topupOrder.update({
          where: { id: order.id },
          data: {
            status: 'FAILED',
            supplierStatus: 'Failed',
          },
        });
      }
      return order;
    });
  }

  const order = await prisma.topupOrder.create({
    data: {
      userId: user.id,
      gameId: game?.id || null,
      gameCode,
      productCode: product.code,
      productName: product.name,
      priceApi,
      priceSell,
      fee,
      totalPay,
      paymentMethod,
      status: 'PENDING_PAYMENT',
      targetUserId: inputs.user_id || inputs.uid || inputs.id || '',
      targetServer: inputs.server || null,
      inputPayload: inputs,
    },
  });

  const paymentPayload = paymentMethod === 'QRIS'
    ? await cashifyService.createQrisV2(totalPay.toNumber())
    : await cashifyService.payEwallet(totalPay.toNumber());

  const paymentId = extractTransactionId(paymentPayload);
  const paymentMeta = extractPaymentMeta(paymentPayload);

  await prisma.topupOrder.update({
    where: { id: order.id },
    data: {
      paymentId,
      inputPayload: {
        ...inputs,
        payment: paymentMeta,
      },
    },
  });

  return {
    order,
    payment: {
      payment_id: paymentId,
      amount: priceSell,
      fee,
      total: totalPay,
      meta: paymentMeta,
    },
  };
}

async function handlePaymentStatus(paymentId) {
  const paymentStatus = await cashifyService.checkStatus(paymentId);
  const status = (paymentStatus?.data?.status || paymentStatus?.status || '').toString().toUpperCase();

  const order = await prisma.topupOrder.findFirst({ where: { paymentId } });
  if (!order) {
    return { status, updated: false };
  }

  if (status === 'PAID' && order.status === 'PENDING_PAYMENT') {
    await prisma.topupOrder.update({
      where: { id: order.id },
      data: { status: 'PAID' },
    });
    try {
      const destination = order.targetUserId;
      const serverId = order.targetServer || '';
      const res = await apigamesService.createTransaction({
        refId: order.id,
        productCode: order.productCode,
        destination,
        serverId,
      });
      const supplierStatus = res?.data?.status || res?.status || 'Pending';
      await prisma.topupOrder.update({
        where: { id: order.id },
        data: {
          status: supplierStatus.toLowerCase() === 'sukses' ? 'SUCCESS' : 'PROCESSING',
          supplierRef: res?.data?.trx_id || res?.data?.ref_id || null,
          supplierStatus,
        },
      });
      return { status, updated: true, orderId: order.id };
    } catch (err) {
      await prisma.topupOrder.update({
        where: { id: order.id },
        data: {
          status: 'FAILED',
          supplierStatus: 'Failed',
        },
      });
      return { status: 'FAILED', updated: true, orderId: order.id };
    }
  }

  if (status === 'EXPIRED') {
    await prisma.topupOrder.update({
      where: { id: order.id },
      data: { status: 'EXPIRED' },
    });
  }

  if (status === 'FAILED') {
    await prisma.topupOrder.update({
      where: { id: order.id },
      data: { status: 'FAILED' },
    });
  }

  if (status === 'CANCELLED' || status === 'CANCELED') {
    await prisma.topupOrder.update({
      where: { id: order.id },
      data: { status: 'CANCELLED' },
    });
  }

  return { status, updated: false, orderId: order.id };
}

async function listOrders(userId) {
  return prisma.topupOrder.findMany({
    where: { userId },
    orderBy: { createdAt: 'desc' },
  });
}

module.exports = {
  createOrder,
  handlePaymentStatus,
  listOrders,
};
