(async function () {
  const token = getToken();
  if (!token) {
    window.location.href = 'login.html';
    return;
  }

  const listContainer = document.getElementById('transaksi-list');
  if (!listContainer) return;

  let allOrders = [];

  try {
    const profile = await apiFetch('/api/me');
    const nameEls = document.querySelectorAll('.text h5');
    if (nameEls && nameEls.length > 1) {
      nameEls[1].textContent = profile.name;
    }

    const [ordersRes, walletRes] = await Promise.all([
      apiFetch('/api/orders').catch(() => ({ data: [] })),
      apiFetch('/api/wallet/transactions').catch(() => ({ data: [] })),
    ]);

    const orders = ordersRes.data || [];
    const walletTx = walletRes.data || [];
    const combined = normalizeTransactions(orders, walletTx);
    allOrders = combined;

    renderOrders(combined);
    setupFilters(combined);
  } catch (err) {
    listContainer.innerHTML = '<p>Gagal memuat transaksi.</p>';
  }

  function normalizeStatus(status) {
    const value = String(status || '').toUpperCase();
    if (value === 'SUCCESS' || value === 'PAID') return 'SUCCESS';
    if (value === 'FAILED' || value === 'EXPIRED' || value === 'CANCELLED' || value === 'CANCELED') return 'FAILED';
    return 'PENDING';
  }

  function mapStatus(item) {
    if (item.status === 'SUCCESS') return { text: 'Sukses', className: 'text-white bg-success text-center rounded-1' };
    if (item.status === 'FAILED') return { text: 'Gagal', className: 'text-white bg-danger text-center rounded-1' };
    return { text: 'Pending', className: 'text-white bg-warning text-center rounded-1' };
  }

  function normalizeTransactions(orders, walletTx) {
    const normalizedOrders = orders.map(order => ({
      id: order.id,
      createdAt: order.createdAt,
      status: normalizeStatus(order.status),
      title: order.productName || 'Top Up Game',
      subtitle: order.gameCode ? `Game: ${order.gameCode}` : 'Top Up',
      amount: order.totalPay,
      rawStatus: order.status,
      method: order.paymentMethod,
    }));

    const normalizedWallet = walletTx.map(tx => {
      const typeLabel = tx.type === 'DEPOSIT'
        ? 'Deposit Saldo'
        : tx.type === 'PURCHASE'
          ? 'Pembelian'
          : tx.type === 'REFUND'
            ? 'Refund'
            : 'Penyesuaian';
      const title = tx.description || typeLabel;
      const subtitle = tx.paymentMethod ? `Metode: ${tx.paymentMethod}` : typeLabel;
      return {
        id: tx.id,
        createdAt: tx.createdAt,
        status: normalizeStatus(tx.status),
        title,
        subtitle,
        amount: tx.total || tx.amount,
        rawStatus: tx.status,
        method: tx.paymentMethod,
      };
    });

    return [...normalizedOrders, ...normalizedWallet].sort((a, b) => {
      const timeA = new Date(a.createdAt).getTime();
      const timeB = new Date(b.createdAt).getTime();
      return timeB - timeA;
    });
  }

  function renderOrders(orders) {
    const total = orders.length;
    const success = orders.filter(o => o.status === 'SUCCESS').length;
    const failed = orders.filter(o => o.status === 'FAILED').length;
    const pending = total - success - failed;

    const totalEl = document.getElementById('gx-total-trx');
    if (totalEl) totalEl.textContent = `${total} Trx`;
    const successEl = document.getElementById('gx-transaksi-sukses');
    if (successEl) successEl.textContent = success;
    const failedEl = document.getElementById('gx-transaksi-gagal');
    if (failedEl) failedEl.textContent = failed;
    const pendingEl = document.getElementById('gx-transaksi-belumdiproses');
    if (pendingEl) pendingEl.textContent = pending;

    listContainer.innerHTML = '';
    if (!orders.length) {
      listContainer.innerHTML = '<div class="text-center"><p>Transaksi tidak ada</p></div>';
      return;
    }

    orders.forEach((order) => {
      const status = mapStatus(order);
      const date = order.createdAt ? new Date(order.createdAt) : null;
      const dateLabel = date ? date.toLocaleDateString('id-ID') : '-';
      const card = document.createElement('div');
      card.innerHTML = `
        <div class="gx-transaksi">
          <div class="d-flex justify-content-between align-items-center border border-secondary-subtle rounded-1 mb-2 animate__animated animate__fadeInLeft transaction-item"
            data-bs-toggle="modal" data-bs-target="#exampleModal"
            data-nomor="${order.id}" data-status="${order.rawStatus || order.status}"
            data-tanggal="${date ? date.toLocaleString('id-ID') : '-'}"
            data-nama="${order.title}" data-url="#" data-link="#"
            data-method="${order.method || ''}"
            style="height:80px;">
            <img class="rounded-1" src="assets/icons/placeholder.svg" style="width:60px;height:60px;margin-left:9px;" />
            <div class="w-100 ms-1">
              <p class="op-7 f-12 fwb-6">${order.title}</p>
              <p class="op-7 f-11">${order.subtitle || order.id}</p>
              <p class="op-7 f-11">${dateLabel}</p>
            </div>
            <p class="fwb-6 f-14 text-end me-1 ${status.className}" style="width:45%">${status.text}</p>
          </div>
        </div>
      `;
      listContainer.appendChild(card);
    });
  }

  function setupFilters(allOrders) {
    const btnAll = document.getElementById('btnSemua');
    const btnSuccess = document.getElementById('btnSukses');
    const btnFailed = document.getElementById('btnGagal');

    if (btnAll) {
      btnAll.addEventListener('click', () => {
        btnAll.classList.add('active');
        btnSuccess?.classList.remove('active');
        btnFailed?.classList.remove('active');
        renderOrders(allOrders);
      });
    }

    if (btnSuccess) {
      btnSuccess.addEventListener('click', () => {
        btnSuccess.classList.add('active');
        btnAll?.classList.remove('active');
        btnFailed?.classList.remove('active');
        renderOrders(allOrders.filter(o => o.status === 'SUCCESS'));
      });
    }

    if (btnFailed) {
      btnFailed.addEventListener('click', () => {
        btnFailed.classList.add('active');
        btnAll?.classList.remove('active');
        btnSuccess?.classList.remove('active');
        renderOrders(allOrders.filter(o => o.status === 'FAILED'));
      });
    }

    const searchInput = document.getElementById('exampleFormControlInput1');
    const clearButton = document.getElementById('clearButton');
    const searchButton = document.getElementById('searchButton');
    const notification = document.getElementById('notification');

    window.searchProducts = function searchProducts() {
      if (!searchInput) return;
      const term = searchInput.value.trim().toLowerCase();
      if (clearButton) {
        clearButton.classList.toggle('hidden', term.length === 0);
      }
      if (searchButton) {
        searchButton.classList.toggle('hidden', term.length > 0);
      }
      if (!term) {
        if (notification) notification.classList.add('hidden');
        renderOrders(allOrders);
        return;
      }
      const filtered = allOrders.filter((order) => {
        const haystack = `${order.title || ''} ${order.subtitle || ''} ${order.id || ''}`.toLowerCase();
        return haystack.includes(term);
      });
      renderOrders(filtered);
      if (notification) {
        notification.classList.toggle('hidden', filtered.length !== 0);
      }
    };

    window.closeSearch = function closeSearch() {
      if (searchInput) searchInput.value = '';
      if (clearButton) clearButton.classList.add('hidden');
      if (searchButton) searchButton.classList.remove('hidden');
      if (notification) notification.classList.add('hidden');
      renderOrders(allOrders);
    };

    const modal = document.getElementById('exampleModal');
    if (modal) {
      modal.addEventListener('show.bs.modal', (event) => {
        const button = event.relatedTarget;
        if (!button) return;
        const status = button.getAttribute('data-status');
        const nomor = button.getAttribute('data-nomor');
        const nama = button.getAttribute('data-nama');
        const tanggal = button.getAttribute('data-tanggal');
        const method = button.getAttribute('data-method');

        const modalStatus = document.getElementById('modalStatus');
        const modalNomor = document.getElementById('modalNomor');
        const modalNama = document.getElementById('modalNama');
        const modalTanggal = document.getElementById('modalTanggal');
        const modalMetode = document.getElementById('modalMetodepembayaran');

        if (modalStatus) modalStatus.textContent = `Status Transaksi ${status}`;
        if (modalNomor) modalNomor.textContent = `#${nomor}`;
        if (modalNama) modalNama.textContent = nama;
        if (modalTanggal) modalTanggal.textContent = tanggal;
        if (modalMetode) modalMetode.textContent = method || '-';
      });
    }
  }
})();
